#ifndef CMEMORYDC_H
#define CMEMORYDC_H

/* The cMemoryDC class is used to maintain a bitmap in memory.
Some uses of cMemoryDC objects are:

* To keep a copy of a CView window to use in restoring the window.
* To use as hidden scratch paper for assembling an image to then BitBlt to the screen.
  This is done either when drawing temporary objects or during animation.
* To use for small bitmaps to represent movable objects such as game characters.

	cMemoryDC uses a constructor that takes a *Cwnd as param and a COLORREF 
backgroundcolor.

	Here are the steps to use a cMemoryDC to maintain a CView window's image.  
Suppose that your application is called My, and you have CMyView class that is a 
child of CView.

(1)	Add a member to your CMyView like this:
		cMemoryDC *pMemDC;

(2)	In your CMyView constructor, put
		pMemDC = new cMemoryDC(this, backgroundcolor);
	backgroundcolor should be RGB(0,0,0) or RGB(255,255,255), depending on
whether your wndclass.hbrBackground is BLACK_BRUSH or WHITE_BRUSH.  
(Alternately you could handle the WM_CREATE message and initialize pMemDC
inside CMyView::OnCreate.  But it seems simpler to do it in the constructor.)

(3)	In your CMyView::OnDraw(CDC *pDC), add these lines

		pDC->GetClipBox(&rect);
		if (!pDC->IsPrinting())
			pMemDC->copyTo(pDC, rect);
		else
			pMemDC->stretchTo(pDC, rect, ST_ISO_USE_WHOLE_SOURCE);

(4)	In your CMyView destructor, put
		delete pMemDC;

(5)	To clear the window from within any CMyView call, (i.e. to fill it with the
cMemoryDC backgroundcolor), put:
		pMemDC->Clear();
		Invalidate(FALSE);

(6)	Use the standard Windows API GUI calls to write graphics to your cMemoryDC.
Whenever you want to make a persistent mark in your CView window, make
the mark in *pMemDC, and then call InvalidateRect.  For instance:
		pMemDC->Rectangle(l, t, r, b);
		Invalidate(FALSE);
*/

/* The meaning of these flags is explained below.  The are used in the last
argument to cMemoryDC::stretchTo and in the adjustStretch helper function. */
#define ST_NON_ISO 1
#define ST_ISO_COVER_TARGET 2
#define ST_ISO_USE_WHOLE_SOURCE 4
#define ST_INTEGER 8

class cMemoryDC : public CDC
{
private:
	CBitmap _cBitmap;
	CBrush _cBrush_blank;
	int _cx, _cy;
public:
//Constructor and Destructor
	cMemoryDC(CWnd *pWnd, COLORREF blankcol = RGB(255, 255, 255)); 
	~cMemoryDC(); //destructor
//Accessors
	int cx(){return _cx;}
	int cy(){return _cy;}
	CRect rect(){return CRect(0, 0, _cx, _cy);}
//Methods
	void clear();
	void copyTo(CDC *pDC, CRect *pRect = NULL);
		//Have this so you can leave out 2nd arg.
	void copyTo(CDC *pDC, const CRect &rect);
	void stretchTo(CDC *pDC, const CRect &rect, int nFlags = ST_NON_ISO);
		 //See comment below on ST_? flags.
};

/*	The ST flags are used in the last argument of stretchTo and are passed
to the adjustStretch helper functions.
	ST_NON_ISO means exactly cover the target rect, even if this means anisotropic stretching.
	ST_ISO_COVER_TARGET and ST_ISO_USE_WHOLE_SOURCE are isotropic stretches.
In ST_ISO_COVER_TARGET we do make sure the stretch covers the entire target area.
In ST_ISO_USE_WHOLE_SOURCE we make sure the stretch fits the entire source area 
into the target area.
	Logically, ST_NON_ISO, ST_ISO_COVER_TARGET, and ST_ISO_USE_WHOLE_SOURCE 
are mutually exclusive, but if you happen to turn on more than one of them, the
priority is in the order they are listed.  And if you turn on none of them,
the default assumption is ST_NON_ISO.
	The additional ST_INTEGER flag has not been thoroughly tested.  It
can be used with any (or none) of the three ST_*_ISO flags, it has the effect of
causing the stretch to be an integral or inverse integral quantity; the reason
you might do this is because integral and inverse integral StretchBlt is 
a faster graphics card operation.  Note that by itself, the default stretchTo
ST_NON_ISO flag does the most obvious thing: it stretches the source exactly
to the target.  If you use ST_NON_ISO|ST_INTEGER, or ST_ISO_USE_WHOLE_SOURCE (with or without ST_INTEGER), you need to resize your target window to make sure 
it is all covered.
	The adjustStretch helper function is used in cMemoryDC::stretchTo.
There's no particular reason to make this function a member of cMemoryDC,
so we make it a global function in case we might want to use it somewhere
else sometime. */

void adjustStretch(int source_cx, int source_cy, int target_cx,
		int target_cy, int &stretch_target_cx, int &stretch_target_cy,
		int _nFlags);

#endif //CMEMORYDC_H
